;;;; FILE parenscript-tagbody-go.lisp

(cl:in-package #:parenscript-tagbody-go)

(defvar *trace-go-tag-info*
        nil
        "If true, then print go-tag-info information
to cl:*trace-output* during (parenscript:ps* '(cl:tagbody ...))
processing.  Also print the information during
(parenscript:ps (cl:tagbody ...)) processing.")

(defun gensymsn (sym)
  (gensym (symbol-name sym)))

(defun symbol-string-comparison (comparator sym-a sym-b)
  (if (eql sym-a sym-b)
      T
      (let
	  ((str-a (format nil "~A::~A"
			  (symbol-package sym-a)
			  (symbol-name sym-a)))
	   (str-b (format nil "~A::~A"
			  (symbol-package sym-b)
			  (symbol-name sym-b))))
	(funcall comparator str-a str-b))))

(defun go-tag= (a b)
  (eql a b))

(defun go-tag< (a b)
    "By convention, integer go-tags
are always less than symbol go-tags.

Symbol go-tags are compared by the strings of their
home packages and symbol names."
  (etypecase a
    (integer
      (etypecase b
	(integer (< a b))
	(symbol T)))
    (symbol
      (etypecase b
	(integer NIL)
	(symbol (symbol-string-comparison #'string< a b))))))

#|
;; (SETF name) in (FLET ...)
(let ((bla 3))
  (flet ((get-bla () bla)
         ((setf get-bla) (new-val) (setf bla new-val)))
    (list bla (get-bla) (setf (get-bla) 4) (get-bla) bla)))

|#
(defmacro with-go-tag-data-structure-helpers (&body body)
    "The helpers are modelled after constructor, predicate,
copier, selectors, mutators, readers, writers and
accessors provided by (cl:defstruct ...).  The
printed representation is that of lists.

Additionally, there is an equality and a strictly-less-than
predicate.  They are (GO-TAG-INFO= a b) and
(GO-TAG-INFO< a b), respectively.

Uses a list-based data structure for the information
associated with go-tags, so that the translation
code can be compiled to other programming languages
more easily.  CL structures and CL classes may not
have nice compilation targets as opposed to a
property list (plist) or an association list (alist)
and the related helpers.

Remember (cl:with-slots ...) and (cl:with-accessors ...).

Fields / Information:

1. GO-TAG: a CL GO tag, so a symbol or an integer
2. SWITCH-VAR: a symbol, usually a gensym
3. BLOCK-VAR: a symbol, usually a gensym
4. CASE-COUNTER: an integer


Examples:
(with-go-tag-data-structure-helpers
  (make-go-tag-info :go-tag 'foo
                    :switch-var (gensym \"switch-\")
                    :block-var (gensym \"block-\")
                    :case-counter 0))

(with-go-tag-data-structure-helpers
  (go-tag-info-p
    (make-go-tag-info :go-tag 'foo
		      :switch-var (gensym \"switch-\")
		      :block-var (gensym \"block-\")
                      :case-counter 0)))

(with-go-tag-data-structure-helpers
  (copy-go-tag-info
    (make-go-tag-info :go-tag 'foo
		      :switch-var (gensym \"switch-\")
		      :block-var (gensym \"block-\")
                      :case-counter 0)))


Constructor: (make-go-tag-info &key go-tag switch-var
                                    block-var case-counter)

Predicate: (go-tag-info-p ...)

Copier: (copy-go-tag-info ...)

Readers / Selectors:
(go-tag-info-go-tag ...)
(go-tag-info-switch-var ...)
(go-tag-info-block-var ...)
(go-tag-info-case-counter ...)

Writers / Mutators:
(setf (go-tag-info-go-tag ...) new-go-tag)
(setf (go-tag-info-switch-var ...) new-switch-var)
(setf (go-tag-info-block-var ...) new-block-var)
(setf (go-tag-info-case-counter ...) new-case-counter)

Accessors:
(go-tag-info-go-tag ...)
(go-tag-info-switch-var ...)
(go-tag-info-block-var ...)
(go-tag-info-case-counter ...)"
  `(labels
       ((make-go-tag-info (&key go-tag switch-var block-var case-counter)
            ""
          (check-type go-tag (or symbol integer))
          (check-type switch-var symbol)
          (check-type block-var symbol)
          (check-type case-counter integer)
          `('go-tag-info :go-tag ,go-tag
                         :switch-var ,switch-var
                         :block-var ,block-var
                         :case-counter ,case-counter))

        (go-tag-info-p (obj)
            "Don't force the order of the plist keys."
          (and (listp obj)
               (eq 'go-tag-info (first obj))
               ;; Quicker check for (= (length obj) 9)
               (let ((tail (nthcdr (1- 9) obj)))
                 (and tail
                      (not (cdr tail))))
               (let ((data-plist (cdr obj)))
                 (and (typep (getf data-plist :go-tag)
		             '(or symbol integer))
		      (typep (getf data-plist :switch-var)
			     'symbol)
		      (typep (getf data-plist :block-var)
			     'symbol)
		      (typep (getf data-plist :case-counter)
			     'integer)))
               ;; Return OBJ upon success for function chaining.
	       obj))

        (go-tag-info= (a b)
          (and (go-tag-info-p a)
	       (go-tag-info-p b)
	       ;; List based structure without nesting.
	       ;;  (equal a b) is sufficient.
	       (equal a b)))

        (go-tag-info< (a b)
            "By convention, integer go-tags
are always less than symbol go-tags.

Symbol go-tags are compared by the strings of their
home packages and symbol names.

The fields are tested in this order:
1. GO-TAG
2. SWITCH-VAR
3. BLOCK-VAR
4. CASE-COUNTER
"
          (check-type a (satisfies go-tag-info-p))
          (check-type b (satisfies go-tag-info-p))
	  (let
	      ((tag-a (go-tag-info-go-tag a))
	       (tag-b (go-tag-info-go-tag b))

	       (sw-a (go-tag-info-switch-var a))
	       (sw-b (go-tag-info-switch-var b))

	       (bl-a (go-tag-info-block-var a))
	       (bl-b (go-tag-info-block-var b))

	       (ctr-a (go-tag-info-case-counter a))
	       (ctr-b (go-tag-info-case-counter b)))
	    (or (go-tag< tag-a tag-b)
		(and (go-tag= tag-a tag-b)
		     (or (symbol-string-comparison #'string<
						   sw-a
						   sw-b)
			 (and (eql sw-a sw-b)
			      (or (symbol-string-comparison #'string<
							    bl-a
							    bl-b)
				  (and (eql bl-a bl-b)
				       (< ctr-a ctr-b)))))))))

        (copy-go-tag-info (go-tag-info)
          (copy-seq go-tag-info))

        (go-tag-info-go-tag (go-tag-info)
          (destructuring-bind
                (_go-tag-info-symbol &key go-tag &allow-other-keys)
              go-tag-info
            (declare (ignore _go-tag-info-symbol))
            go-tag))

        ((setf go-tag-info-go-tag) (new-go-tag go-tag-info)
          (destructuring-bind
                (_go-tag-info-symbol &key go-tag &allow-other-keys)
              go-tag-info
            (declare (ignore _go-tag-info-symbol))
            (setf go-tag new-go-tag)))

        (go-tag-info-switch-var (go-tag-info)
          (destructuring-bind
                (_go-tag-info-symbol &key switch-var &allow-other-keys)
              go-tag-info
            (declare (ignore _go-tag-info-symbol))
            switch-var))

        ((setf go-tag-info-switch-var) (new-switch-var go-tag-info)
          (destructuring-bind
                (_go-tag-info-symbol &key switch-var &allow-other-keys)
              go-tag-info
            (declare (ignore _go-tag-info-symbol))
            (setf switch-var new-switch-var)))

        (go-tag-info-block-var (go-tag-info)
          (destructuring-bind
                (_go-tag-info-symbol &key block-var &allow-other-keys)
              go-tag-info
            (declare (ignore _go-tag-info-symbol))
            block-var))

        ((setf go-tag-info-block-var) (new-block-var go-tag-info)
          (destructuring-bind
                (_go-tag-info-symbol &key block-var &allow-other-keys)
              go-tag-info
            (declare (ignore _go-tag-info-symbol))
            (setf block-var new-block-var)))

        (go-tag-info-case-counter (go-tag-info)
          (destructuring-bind
                (_go-tag-info-symbol &key case-counter &allow-other-keys)
              go-tag-info
            (declare (ignore _go-tag-info-symbol))
            case-counter))

        ((setf go-tag-info-case-counter) (new-case-counter go-tag-info)
           (destructuring-bind
                 (_go-tag-info-symbol &key case-counter &allow-other-keys)
               go-tag-info
             (declare (ignore _go-tag-info-symbol))
             (setf case-counter new-case-counter))))
     ,@body))



;;; ===============================
;;; 2022-02-24

;; Parenscript example:

#+()
(let ((outer-block-1 (gensymsn 'outer-block-1-))
      (preamble-block-1 (gensymsn 'preamble-block-1-))
      (inner-block-1 (gensymsn 'inner-block-1-))
      (switch-var-1 (gensymsn 'switch-var-1-))
      (outer-block-2 (gensymsn 'outer-block-2-))
      (preamble-block-2 (gensymsn 'preamble-block-2-))
      (inner-block-2 (gensymsn 'inner-block-2-)))
  `(block ,outer-block-1
     (let ((,switch-var-1 tagbody-1-first-tag))
       (block ,preamble-block-1
         (preamble-1-call-1)
	 ;; preamble jump (go tagbody-1-tag-2)
         (progn
	   (setf ,switch-var-1 'tagbody-1-tag-2)
	   (return-from ,preamble-block-1))
         (preamble-1-call-2))
       (loop do
         (block ,inner-block-1
	   (parenscript:switch ,switch-var-1
	     (case tagbody-1-tag-1
	       (foo)
	       (block ,outer-block-2
		 (let ((,switch-var-2 tagbody-2-first-tag))
                   (block ,preamble-block-2
		     (preamble-2))
		   (loop do
		     (block ,inner-block-2
		       (parenscript:switch ,switch-var-2
	                 (case tagbody-2-tag-1)
			   ;; inner jump: (go tagbody-2-tag-2)
                           (progn
			     (setf ,switch-var-2 'tagbody-2-tag-2)
			     (return-from ,inner-block-2))
			   ;; outer jump: (go tagbody-1-tag-2)
                           (progn
			     (setf ,switch-var-1 'tagbody-1-tag-2)
			     (return-from ,inner-block-1))
	                 (case tagbody-2-tag-2)
                           ;; Walking off the end of tagbody-2
		           (return-from ,outer-block-2))))))
               ;; Code to skip when jumping from the
	       ;;  inner tagbody to a go tag in the
	       ;;  outer tagbody. Nevertheless, it has
	       ;;  to be run, when walking off the end of
	       ;;  the inner tagbody.
	       (bar))
	       (case tagbody-1-tag-2
		 (baz)
                 ;; Walking off the end of tagbody-1
		 (return-from ,outer-block-1))))))))





;;; ===============================
;;; 2022-03-19

;; Needs to be a lisp instead of a parenscript macro,
;;  because (parenscript:defpsmacro ...) is a
;;  lisp form instead of a parenscript form.
(defmacro with-tagbody-helpers (&body body)
  `(labels
       ((go-tag-p (obj)
            "As by CLHS valid go-tags are symbols or integers.
             Recall, that as per the CLHS,
             no macroexpansion or symbol-macroexpansion
             is to be performed *before* deciding whether
             any given top-level object within a (tagbody ...)
             form is a go-tag, or not."
          (or (symbolp obj) (integerp obj)))

        (tb-go-tags (tb-body)
            "Returns a list of go-tags in a given body of
             a (tagbody ...) form."
          (remove-if-not #'go-tag-p tb-body))

        (first-go-tag (tb-body)
	  ;; Find-if does *not* work cleanly.  It fails
	  ;;  to distinguish between a tag named nil
	  ;;  and the absence of go tags.  The latter
	  ;;  is solely having a preamble in the
	  ;;  tagbody form.
	  "Returns two values like CL:GETHASH.
	  1. First tag.
	  2. Whether a tag was found. Relevant in case
	  the first return value is NIL.

	  Note, that NIL is a valid go-tag."
	  (block first-go-tag
	    (loop for form in tb-body
	      do (if (go-tag-p form)
		   (return-from first-go-tag
		     (values form t))))
            (return-from first-go-tag
	      (values nil nil))))
  
        (split-and-group-tb-body (tb-body)
          "Returns two values.
          1. The preamble -- code without a preceding tag
          2. Grouping of tags and subsequent code."
 
	  (block split-and-group-tb-body
            (if (null tb-body)
	      (return-from split-and-group-tb-body
		           (values nil nil)))
            (let ((acc `((,(first tb-body))))
	          (preamble-p (not (go-tag-p (first tb-body)))))
	      (loop for tbf in (rest tb-body) do
	        (if (go-tag-p tbf)
	          (push `(,tbf) acc)
	          (push tbf (first acc))))
              (setf acc (nreverse (mapcar #'nreverse acc)))
	      (if preamble-p
	        (values (first acc) (rest acc))
	        (values nil acc))))))
     ,@body))




;; Needs to be a lisp instead of a parenscript macro,
;;  because (parenscript:defpsmacro ...) is a
;;  lisp form instead of a parenscript form.
(defmacro with-tagbody-parenscript-helpers (&body body)
  `(with-tagbody-helpers
     (labels
	 ((new-go-bindings (switch-var block-var last-case-counter
                            new-tb-go-tags)
              "Collect up all information necessary to
               implement (go ...) forms.  Also deals with
               nested (tagbody ...) forms."
	    (mapcar (lambda (go-tag case-counter)
                      ;; List of list structures.  Use similar to an alist.
                      ;;  See (with-go-tag-data-structure-helpers ...).
                      ;;  See also (apropos "go-tag-info" #:parenscript-tagbody-go)
                      (with-go-tag-data-structure-helpers
                        (make-go-tag-info :go-tag go-tag
		  			  :switch-var switch-var
		  			  :block-var block-var
		  			  :case-counter case-counter)))
		    new-tb-go-tags
		    (loop for i from (1+ last-case-counter)
                          repeat (length new-tb-go-tags)
			  collect i)))
	 (grouping-to-case-forms (grouped-tb-body
                                  new-case-counter
				  old-and-new-go-bindings-for-rest-of-tbbody)
              "Group go-tags and the following non-go-tag forms."
	    (mapcar (lambda (go-tag-case)
	              (destructuring-bind
		            (go-tag &rest case-body)
			  go-tag-case
			`(,(with-go-tag-data-structure-helpers ; ,go-tag
			     (go-tag-info-case-counter
			       (find go-tag
				     ;; Sorted newest to oldest.
				     old-and-new-go-bindings-for-rest-of-tbbody
				     :key #'go-tag-info-go-tag)))
		           ,(format nil "tagbody-go-tag: ~A" go-tag)
		           ;; Handle nested tagbody
		           ;;  forms correctly.
		           (tagbody-recursive (,old-and-new-go-bindings-for-rest-of-tbbody
					       ,new-case-counter)
		             ,@case-body))))
	            grouped-tb-body))

	  (tb-body-to-switch (outer-block-var
                              preamble-block-var
	                      inner-block-var
			      preamble
                              grouped-tb-body
			      first-tag
			      switch-var
			      new-case-counter
			      old-and-new-go-bindings-for-preamble
			      old-and-new-go-bindings-for-rest-of-tbbody)
              "Turn a (tagbody ...) body into a (loop (switch ...)).
               Facilitate compilation of nested (tagbody ...) forms."
            (cond
              ((and (null preamble)
		    (null grouped-tb-body))
	       `())
              ((and (not (null preamble))
		    (null grouped-tb-body))
               `(progn ,@preamble))
              ((not (null grouped-tb-body))
	       `(block ,outer-block-var
		  (parenscript:let
		      ((,switch-var ,(with-go-tag-data-structure-helpers
				       (go-tag-info-case-counter
					 (find first-tag
					       ;; Sorted newest to oldest.
					       old-and-new-go-bindings-for-rest-of-tbbody
					       :key #'go-tag-info-go-tag)))))
		    (block ,preamble-block-var
			   (parenscript:macrolet
			     ((go (go-tag)
				  (with-go-tag-data-structure-helpers
				    (let ((gti (find go-tag
						     ',old-and-new-go-bindings-for-preamble
						     :key #'go-tag-info-go-tag)))
				      `(progn
					 ,(format nil "GOTO/JMP tagbody-go-tag: ~A" go-tag)
					 (setf ,(go-tag-info-switch-var gti)
					       ,(go-tag-info-case-counter gti))
					 ,(format nil "GOTO/JMP tagbody-go-tag: ~A" go-tag)
					 (return-from ,(go-tag-info-block-var gti)))))))
			     (tagbody-recursive (,old-and-new-go-bindings-for-preamble
						  ,new-case-counter)
						,@preamble)))
		    (loop do
			  (block ,inner-block-var
				 (parenscript:macrolet
				   ((go (go-tag)
					(with-go-tag-data-structure-helpers
					  (let ((gti (find go-tag
							   ',old-and-new-go-bindings-for-rest-of-tbbody
							   :key #'go-tag-info-go-tag)))
					    `(progn
					       ,(format nil "GOTO/JMP tagbody-go-tag: ~A" go-tag)
					       (setf ,(go-tag-info-switch-var gti)
						     ,(go-tag-info-case-counter gti))
					       (return-from ,(go-tag-info-block-var gti)))))))
				   (parenscript:switch ,switch-var
						       ,@(grouping-to-case-forms
							   grouped-tb-body
							   new-case-counter
							   old-and-new-go-bindings-for-rest-of-tbbody)))
				 ;; Fall-through after end of tagbody form
				 (return-from ,outer-block-var)))))))))
,@body)))



;; Needs to be a parenscript macro, because it
;;  create a lexical macro-binding for (tagbody ...)
;;  which is supposed to be macroexpanded
;;  by parenscript's compiler.
(parenscript:defpsmacro
    tagbody-recursive
    ((&optional outer-go-bindings
      (last-case-counter 0))
  &body body)
"Recursion information OUTER-GO-BINDINGS only by
nested calls.  Confer recursion flag of #'CL:READ.

Also known as: with-tagbody-for-parenscript-recursive."
  (when *trace-go-tag-info*
    (format *trace-output*
	    "TOP-LEVEL PSMACRO (TAGBODY-RECURSIVE) BODY: ~S~%"
	    body))
  (if (null body)
      nil
      (let ((outer-block-var (gensymsn 'outer-block-var-))
	    (preamble-block-var (gensymsn 'preamble-block-var-))
	    (inner-block-var (gensymsn 'inner-block-var-))
	    (switch-var (gensymsn 'switch-var-)))
	`(parenscript:macrolet
	   ((tagbody (&body tb-body)
	      (when *trace-go-tag-info*
		(format *trace-output* "NESTED (TAGBODY) TB-BODY: ~S~%" tb-body))
	      (if (null tb-body)
		  nil
		  (with-tagbody-parenscript-helpers
		    (let* ((new-go-tags (tb-go-tags tb-body))
			   (first-go-tag (first-go-tag tb-body))
			   (old-and-new-go-bindings-for-preamble
			     ;; list of go-tag-info lists
			     (append
			       (new-go-bindings ',switch-var
						',preamble-block-var
						;; Self-evaluating integer.
						;;  No quoting necessary.
						,last-case-counter
						new-go-tags)
			       ',outer-go-bindings))
			   (old-and-new-go-bindings-for-rest-of-tbbody
			     ;; list of go-tag-info lists
			     (append
			       (new-go-bindings ',switch-var
						',inner-block-var
						;; Self-evaluating integer.
						;;  No quoting necessary.
						,last-case-counter
						new-go-tags)
			       ',outer-go-bindings)))
		      (when *trace-go-tag-info*
			;; DEBUG go-tag analysis and translation
			(format *trace-output* "new go tags: ~S~%" new-go-tags)
			(format *trace-output* "first go tag: ~S~%" first-go-tag)
			(format *trace-output* "old and new bindings for preamble: ~S~%" old-and-new-go-bindings-for-preamble)
			(format *trace-output* "old and new bindings for rest of tbbody: ~S~%" old-and-new-go-bindings-for-rest-of-tbbody))
                      ;; Check for duplicate tags in (tagbody ...) form.
                      ;;  Error out if there are duplicate tags.
                      (let*
                          ((sorted-tags (sort (copy-seq new-go-tags)
					      #'go-tag<))
			   (prev-go-tag (first sorted-tags)))
			(loop for gtag in (rest sorted-tags)
			      do (when (go-tag= prev-go-tag gtag)
				   (error "Error: Duplicate jump tag in (tagbody ...) form: ~S" gtag))
			         (setf prev-go-tag gtag)))
		      (multiple-value-bind
			    (preamble tb-groups)
			  (split-and-group-tb-body tb-body)
			(tb-body-to-switch ',outer-block-var
					   ',preamble-block-var
					   ',inner-block-var
					   preamble
					   tb-groups
					   first-go-tag
					   ',switch-var
					   ;; Self-evaluating integer.
					   ;;  No quoting necessary.
					   (+ ,last-case-counter
					      (length new-go-tags))
					   old-and-new-go-bindings-for-preamble
					   old-and-new-go-bindings-for-rest-of-tbbody)))))))
	   ,@body))))

;; Needs to be a parenscript macro, because it
;;  create a macro-binding for (tagbody ...)
;;  which is supposed to be macroexpanded
;;  by parenscript's compiler.
(parenscript:defpsmacro tagbody (&body tb-body)
  (when *trace-go-tag-info*
    (format *trace-output* "TOP-LEVEL (TAGBODY) TB-BODY: ~S~%" tb-body))
  (if (null tb-body)
      nil
      `(tagbody-recursive
	   () ; empty tagbody bindings
	 ;; lexical (tagbody ...) form
	 ;;  established by (parenscript:macrolet ...)
	 (tagbody
	  ,@tb-body))))

;;;; END OF FILE parenscript-tagbody-go.lisp
